/**
 * Service Worker pour FacturDash PWA
 * Version: 1.0.0
 */

const CACHE_NAME = 'facturdash-v1.0.0';
const urlsToCache = [
  '/crm/presentation/',
  '/crm/presentation/index.html',
  '/crm/presentation/download.php',
  '/crm/presentation/manifest.json',
  'https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap',
  'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css'
];

// Installation du Service Worker
self.addEventListener('install', function(event) {
  console.log('Service Worker: Installation en cours...');
  
  event.waitUntil(
    caches.open(CACHE_NAME)
      .then(function(cache) {
        console.log('Service Worker: Cache ouvert');
        return cache.addAll(urlsToCache);
      })
      .then(function() {
        console.log('Service Worker: Installation terminée');
        return self.skipWaiting();
      })
      .catch(function(error) {
        console.error('Service Worker: Erreur lors de l\'installation', error);
      })
  );
});

// Activation du Service Worker
self.addEventListener('activate', function(event) {
  console.log('Service Worker: Activation en cours...');
  
  event.waitUntil(
    caches.keys().then(function(cacheNames) {
      return Promise.all(
        cacheNames.map(function(cacheName) {
          if (cacheName !== CACHE_NAME) {
            console.log('Service Worker: Suppression de l\'ancien cache', cacheName);
            return caches.delete(cacheName);
          }
        })
      );
    }).then(function() {
      console.log('Service Worker: Activation terminée');
      return self.clients.claim();
    })
  );
});

// Interception des requêtes
self.addEventListener('fetch', function(event) {
  console.log('Service Worker: Requête interceptée', event.request.url);
  
  event.respondWith(
    caches.match(event.request)
      .then(function(response) {
        // Retourner la réponse du cache si elle existe
        if (response) {
          console.log('Service Worker: Réponse du cache', event.request.url);
          return response;
        }
        
        // Sinon, faire la requête réseau
        console.log('Service Worker: Requête réseau', event.request.url);
        return fetch(event.request).then(function(response) {
          // Vérifier si la réponse est valide
          if (!response || response.status !== 200 || response.type !== 'basic') {
            return response;
          }
          
          // Cloner la réponse pour le cache
          var responseToCache = response.clone();
          
          caches.open(CACHE_NAME)
            .then(function(cache) {
              cache.put(event.request, responseToCache);
            });
          
          return response;
        }).catch(function(error) {
          console.error('Service Worker: Erreur de requête', error);
          
          // Retourner une page d'erreur personnalisée si c'est une page HTML
          if (event.request.headers.get('accept').includes('text/html')) {
            return caches.match('/crm/presentation/index.html');
          }
        });
      })
  );
});

// Gestion des messages du client
self.addEventListener('message', function(event) {
  console.log('Service Worker: Message reçu', event.data);
  
  if (event.data && event.data.type === 'SKIP_WAITING') {
    self.skipWaiting();
  }
  
  if (event.data && event.data.type === 'GET_VERSION') {
    event.ports[0].postMessage({
      version: CACHE_NAME
    });
  }
});

// Gestion des notifications push (optionnel)
self.addEventListener('push', function(event) {
  console.log('Service Worker: Notification push reçue');
  
  const options = {
    body: event.data ? event.data.text() : 'Nouvelle mise à jour disponible !',
    icon: '/crm/presentation/icons/icon-192x192.png',
    badge: '/crm/presentation/icons/icon-72x72.png',
    vibrate: [100, 50, 100],
    data: {
      dateOfArrival: Date.now(),
      primaryKey: 1
    },
    actions: [
      {
        action: 'explore',
        title: 'Voir les détails',
        icon: '/crm/presentation/icons/icon-96x96.png'
      },
      {
        action: 'close',
        title: 'Fermer',
        icon: '/crm/presentation/icons/icon-96x96.png'
      }
    ]
  };
  
  event.waitUntil(
    self.registration.showNotification('FacturDash', options)
  );
});

// Gestion des clics sur les notifications
self.addEventListener('notificationclick', function(event) {
  console.log('Service Worker: Clic sur notification', event.action);
  
  event.notification.close();
  
  if (event.action === 'explore') {
    event.waitUntil(
      clients.openWindow('/crm/presentation/')
    );
  }
});

console.log('Service Worker: Chargé et prêt');
